<?php

namespace R6Digital\KitchenSink;

class Authentication
{
    public const CLIENT_ID = 'r6digital';
    public const CLIENT_SECRET = 'Dd6gwIWUXYGC2XpzZITzSvD8pK4ggwT5hBkOl6ZB';
    public const SCOPE = 'openid profile email';

    public const AUTHORIZATION_ENDPOINT = 'https://oauth.r6internal.com/oauth/authorize';
    public const TOKEN_ENDPOINT = 'https://oauth.r6internal.com/oauth/access_token';
    public const USER_INFO_ENDPOINT = 'https://oauth.r6internal.com/oauth/user_info';

    public const ENCRYPTION_KEY = 'VJ3e9UGQyqKj32aRDK8QAITC7oz2vcwNmR50100ltEc=';

    public const LOGIN_URL = '/r6-login';

    public function __construct()
    {
        if (self::should_show_login_link()) {
            add_action('login_form', [$this, 'add_login_link']);
            add_action('login_head', [$this, 'add_login_css'], 0);
        }

        if (self::is_enabled()) {
            add_filter('wp_headers', [$this, 'add_r6_login_header']);
        }

        add_action('init', [$this, 'intercept_login_url']);
    }

    public static function is_enabled(): bool
    {
        return get_option(R6_KITCHEN_SINK_ID . '_authentication_enabled', true);
    }

    public static function should_show_login_link(): bool
    {
        return self::is_enabled() && get_option(R6_KITCHEN_SINK_ID . '_authentication_show_login_link', true) && !Compatibility::is_wordpress_social_login_active();
    }

    public static function should_create_new_users(): bool
    {
        return get_option(R6_KITCHEN_SINK_ID . '_authentication_create_new_users', true);
    }

    public function add_login_link() {

        ?>
        <div class="r6-login">
            <div class="r6-login__label">
                Employee login:
            </div>
            <div class="r6-login__link">
                <a href="<?= esc_attr(home_url(self::LOGIN_URL)) ?>">R6</a>
            </div>
        </div>
        <?php
    }

    public function add_login_css() {
        ?>
        <style>
            .r6-login .r6-login__link {
                padding: 10px;
                margin-bottom: 5px;
            }
            .r6-login .r6-login__link a {
                text-decoration: none;
            }
        </style>
        <?php
    }

    public function intercept_login_url()
    {
        if (rtrim(parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH), '/') !== self::LOGIN_URL) {
            return;
        }

        if (!self::is_enabled()) {
            wp_die('The R6 Login has been disabled. An alternative method of authentication is required.', '', [
                    'response' => 403,
                    'link_url' => wp_login_url(),
                    'link_text' => 'Go to login form'
            ]);
        }

        if (isset($_GET['code'])) {
            $this->process_oauth_response();
        } else {
            $this->redirect_to_authorization_endpoint(!empty($_GET['test']));
        }
    }

    public function process_oauth_response()
    {
        $state = $this->verify_state($_GET['state']);
        if (!$state) {
            wp_die('Unable to authenticate: State was invalid.');
        }

        $access_token = $this->get_access_token($_GET['code']);

        if (!$access_token) {
            wp_die('Unable to authenticate: Access token not provided.');
        }

        $user_info = $this->get_user_info($access_token);

        if (!$user_info || empty($user_info['sub'])) {
            echo '<pre>' . print_r($user_info, true) . '</pre>';
            wp_die('Unable to authenticate: Invalid response from user_info endpoint.');
        } elseif ($state['test']) {
            echo '<pre>' . print_r($user_info, true);
            exit;
        } else {
            $this->login_as_user($user_info);
        }
    }

    public function login_as_user($user_info)
    {
        $user = get_user_by('email', $user_info['email']);

        if (!$user && !self::should_create_new_users()) {
            wp_die('Unable to authenticate: Creating new users through R6 Login has been disabled.', '', [
                'response' => 403,
                'link_url' => wp_login_url(),
                'link_text' => 'Go to login form'
            ]);
        }

        $user_array = [
            'first_name' => $user_info['given_name'],
            'last_name' => $user_info['family_name'],
            'user_email' => $user_info['email'],
            'user_login' => str_replace(' ', '_', $user_info['name']),
            'role' => 'administrator',
        ];

        if ($user) {
            $user_array['ID'] = $user->ID;
            $user_id = wp_update_user($user_array);
        } else {
            $user_array['user_pass'] = wp_generate_password();
            $user_id = wp_insert_user($user_array);
        }

        if (is_multisite()) {
            grant_super_admin($user_id);
        }

        wp_set_current_user($user_id);
        wp_set_auth_cookie($user_id);
        $user = get_user_by('ID', $user_id);

        do_action('wp_login', $user->user_login, $user);

        /*
         * Add a random query string parameter to the URL when redirecting back.
         *
         * When a logged out user tries to access wp-admin and is redirected to the login page,
         * browsers can cache that redirect (even though it's a 302!) which will cause the cookies
         * not to be set correctly when we redirect back, which will cause the login to fail.
         * This can be avoided by simply redirecting back to a different URL.
         */
        wp_redirect(admin_url() . '?nonce' . md5(time()), 302, 'Successful login');
        exit;
    }

    /**
     * @param bool $test
     *
     * @return never-return
     */
    public function redirect_to_authorization_endpoint(bool $test = false)
    {
        $state = self::generate_state();
        $state['test'] = $test;

        $params = [
            'client_id' => self::CLIENT_ID,
            'scope' => self::SCOPE,
            'redirect_uri' => home_url(self::LOGIN_URL),
            'response_type' => 'code',
            'state' => base64_encode(json_encode($state))
        ];

        $authorizationUrl = self::AUTHORIZATION_ENDPOINT . '?' . http_build_query($params);

        wp_redirect($authorizationUrl, 302, R6_KITCHEN_SINK_NAME);
        exit;
    }

    public function get_access_token(string $code) {
        $client_id = self::CLIENT_ID;
        $client_secret = self::CLIENT_SECRET;

        $authorization = base64_encode("$client_id:$client_secret");

        $params = [
            'client_id' => $client_id,
            'client_secret' => $client_secret,
            'grant_type' => 'authorization_code',
            'code' => $code,
            'redirect_uri' => home_url(self::LOGIN_URL)
        ];

        $result_full = wp_remote_post(
            self::TOKEN_ENDPOINT,
            [
                'method' => 'POST',
                'headers' => [
                    'Authorization' => 'Basic' . $authorization,
                    'Accept' => 'application/json',
                ],
                'body' => http_build_query($params),
            ]
        );


        // Just the "body" bit
        $result_entities = json_decode($result_full['body'], true);
        return $result_entities['access_token'];
    }

    public function get_user_info(string $access_token)
    {
        $response = wp_remote_post(
            self::USER_INFO_ENDPOINT,
            [
                'method' => 'GET',
                'headers' => [
                    'Authorization' => 'Bearer ' . $access_token
                ]
            ]
        );

        return json_decode($response['body'], true);
    }


    /** @noinspection PhpUnhandledExceptionInspection */
    public function generate_state(): array
    {
        $key = sodium_base642bin(self::ENCRYPTION_KEY, SODIUM_BASE64_VARIANT_ORIGINAL);
        $nonce = random_bytes(SODIUM_CRYPTO_SECRETBOX_NONCEBYTES);

        $plaintext = SECURE_AUTH_KEY;

        return [
            'auth_key' => sodium_bin2hex(sodium_crypto_secretbox($plaintext, $nonce, $key)),
            'nonce' => sodium_bin2hex($nonce),
        ];
    }

    /** @noinspection PhpUnhandledExceptionInspection */
    public function verify_state($state_string)
    {
        $state = json_decode(base64_decode($state_string), true);

        $key = sodium_base642bin(self::ENCRYPTION_KEY, SODIUM_BASE64_VARIANT_ORIGINAL);
        $nonce = sodium_hex2bin($state['nonce']);
        $plaintext = sodium_crypto_secretbox_open(sodium_hex2bin($state['auth_key']), $nonce, $key);

        if ($plaintext !== SECURE_AUTH_KEY) {
            return false;
        }

        return $state;
    }

    private function get_return_url()
    {
        return admin_url();
    }

    public function add_r6_login_header($headers)
    {
        if (!empty($_SERVER['HTTP_X_R6_LOGIN'])) {
            $headers['X-R6-Login-Available'] = 1;
        }

        return $headers;
    }
}
