<?php

namespace R6Digital\KitchenSink;

class UrlParameterTracking
{
    public static $recognised_fields = [
        'utm_source',
        'utm_medium',
        'utm_campaign',
        'utm_term',
        'gclid'
    ];

    public function __construct()
    {
        // Return early if this feature is not enabled.
        if ( ! self::is_enabled()) {
            return;
        }

        // Set the class's list of accepted URL Parameters if a list has been provided by a plugin end-user.
        $tracked_parameters_override = self::get_tracked_parameters_option_values();
        if ($tracked_parameters_override) {
            self::$recognised_fields = $tracked_parameters_override;
        }

        /** @uses capture_url_parameters() */
        add_action('init', [$this, 'capture_url_parameters'], 11);

        /**
         * Filter rocket_cache_ignored_parameters parameters
         *
         * @uses define_ignored_parameters()
         */
        add_filter('rocket_cache_ignored_parameters', [$this, 'define_ignored_parameters']);

        // Enable CF7 integration if the user has enabled the integration.
        if (self::is_cf7_integration_enabled()) {
            add_filter('wpcf7_before_send_mail', [$this, 'append_tracked_parameters_to_cf7_submission'], 20, 3);
        }

        // Enable and provide form integration if the user has provided text to the selector input field.
        if (self::is_input_submitter_enabled()) {
            add_filter('wp_footer', [$this, 'render_input_submitter_element'], 20);
        }
    }

    public static function is_enabled(): bool
    {
        return get_option(R6_KITCHEN_SINK_ID . '_url_parameter_tracking_enabled', false);
    }

    /**
     * Saves any incoming accepted URL Parameters to cookies.
     */
    public function capture_url_parameters(): void
    {
        // If not a valid front-end page for capturing cookies, return early.
        if (
            is_admin()
            || $GLOBALS['pagenow'] === 'wp-login.php'
            || defined('DOING_CRON')
        ) {
            return;
        }

        // Prepare domain name for usage for cookie source, with the identifying proceeding period.
        $domain = $_SERVER["SERVER_NAME"] ?? '';

        if (strtolower(substr($domain, 0, 4)) == 'www.') {
            $domain = substr($domain, 4);
        }

        if (
            substr($domain, 0, 1) != '.'
            && $domain != "localhost"
        ) {
            $domain = '.' . $domain;
        }

        // Save each corresponding URL Parameter to the corresponding cookie.
        foreach (self::$recognised_fields as $field) {
            if (empty($_GET[$field])) {
                continue;
            }

            $cookie_field = htmlspecialchars($_GET[$field], ENT_QUOTES);

            setcookie($field, $cookie_field, time() + 60 * 60 * 24 * 30, '/', $domain);

            $_COOKIE[$field] = $cookie_field;
        }
    }

    public function define_ignored_parameters(array $params): array
    {
        foreach (self::$recognised_fields as $field) {
            unset($params[$field]);
        }

        return $params;
    }

    /**
     * Returns an array of the corresponding URL Parameter values, prefering values immediately submitted via
     * the current URL. If not present, it defaults to the most recent cookie of that URL Parameter if it has been set.
     *
     * @return array
     */
    public static function get_tracked_url_parameter_values(): array
    {
        $tracking_codes = [];
        foreach (self::$recognised_fields as $field) {
            if ( ! empty($_GET[$field])) {
                $tracking_codes[$field] = $_GET[$field];
            } elseif ( ! empty($_COOKIE[$field])) {
                $tracking_codes[$field] = $_COOKIE[$field];
            }
        }

        return $tracking_codes;
    }

    /**
     * Returns the list of specified URL Parameters to be tracked from the WPDB as a usable array.
     *
     * @return array|null
     */
    public static function get_tracked_parameters_option_values(): ?array
    {
        $parameters_string = get_option(R6_KITCHEN_SINK_ID . '_url_parameter_tracking_parameters', true);

        if ($parameters_string && $parameters_string !== 0) {
            return preg_split(
                "/(,|\r\n|\s)/",
                $parameters_string
            );
        } else {
            return null;
        }
    }

    /**
     * Determines if a user has opted the site into appending CF7 integration.
     *
     * @return bool
     */
    public static function is_cf7_integration_enabled(): bool
    {
        return get_option(R6_KITCHEN_SINK_ID . '_url_parameter_tracking_cf7_integration', true) && class_exists('WPCF7_ContactForm');
    }

    /**
     * Appends URL Parameter values to CF7 submission based on mail type (HTML vs Non-HTML).
     *
     * @param \WPCF7_ContactForm $contact_form
     * @param $abort
     * @param $submission
     *
     * @return \WPCF7_ContactForm
     */
    public function append_tracked_parameters_to_cf7_submission(\WPCF7_ContactForm $contact_form, &$abort, $submission): \WPCF7_ContactForm
    {
        $mail = $contact_form->prop('mail');

        if ($mail['use_html']) {
            $mail['body'] .= '<p>';
            foreach (self::get_tracked_url_parameter_values() as $field => $value) {
                $mail['body'] .= "$field: $value <br>";
            }
            $mail['body'] .= '</p>';
        } else {
            $mail['body'] .= PHP_EOL;
            foreach (self::get_tracked_url_parameter_values() as $field => $value) {
                $mail['body'] .= "$field: $value" . PHP_EOL;
            }
        }

        $contact_form->set_properties(['mail' => $mail]);
        error_log(print_r($contact_form, true));

        return $contact_form;
    }

    /**
     * Determines if a user has submitted a selector to specify outlier form fields that require
     * URL Parameter input.
     *
     * @return bool
     */
    public function is_input_submitter_enabled(): bool {
        return get_option(R6_KITCHEN_SINK_ID . '_url_parameter_tracking_form_selector', null);
    }

    /**
     * Creates a render-ready HTML prepared <script type="text/javascript></script> element.
     *
     * This script element is responsible for submitting URL parameters saved as
     * cookies to input fields based on a selector provided by an end-user of the
     * R6 Kitchen Sink plugin (via the R6 Kitchen Sink settings page).
     *
     * If the substring "URL_PARAM" is found, the corresponding URL Parameter's identifier will
     * replace this substring. If "URL_PARAM" is not a substring, the URL Parameter's identifier will
     * instead be appended to the end of the selector.
     *
     * @return string
     */
    public function assemble_input_submitter_element(): string
    {
        // Prepare the $recognised_fields array for rendering in a JavaScript-interoperable format.
        $javascriptUrlParametersArray = '['.implode(
                ',',
                array_map(
                    function ($element) {
                        return "'$element'";
                    },
                    self::$recognised_fields
                )
            ).']';

        $cookieValues = json_encode(self::get_tracked_url_parameter_values());

        // Prepare the form_selector option's value for display in a JavaScript-interoperable format.
        $selector = "'".str_replace('\"', '"', get_option(R6_KITCHEN_SINK_ID . '_url_parameter_tracking_form_selector'))."'";

        return <<<HTML
<script type="text/javascript" defer>
    let trackedParameters = {$javascriptUrlParametersArray};
    let selector = {$selector};
    
    // Initialise from cookies if they exist
    let cookies = document.cookie.split('; ');
    let cookieKeyedValues = new Map(Object.entries({$cookieValues}));
    cookies.forEach(function(cookie) {
        let keyValuePair = cookie.split('=');
        cookieKeyedValues[keyValuePair[0]] = keyValuePair[1];
    });
    
    // Update from URLSearchParams
    let urlParams = new URLSearchParams(window.location.search); 
    urlParams.forEach(function(value, key) {
        cookieKeyedValues[key] = value;
    })
    
    // noinspection JSUnusedAssignment
    trackedParameters.forEach(function(parameter) {
        let fieldSelector = selector;
        
        if (fieldSelector.includes('URL_PARAM')) {
            fieldSelector = fieldSelector.replaceAll('URL_PARAM', parameter);
        } else {
            fieldSelector += parameter;
        }
        
        let fields = document.querySelectorAll(fieldSelector);
        fields.forEach(function(field) {
            field.value = cookieKeyedValues[parameter] ?? '';
        });
    });
</script>
HTML;
    }

    /**
     * Echos the <script></script> element produced by assemble_input_submitter_element.
     */
    public function render_input_submitter_element(): void {
        echo self::assemble_input_submitter_element();
    }
}
