<?php

namespace R6Digital\KitchenSink\WPCF7;

class SitelinkService extends Service
{
    const service_name = 'sitelink';

    const service_url = 'https://cfservers.com/RSlib/reportingLib/custom/API/LeadToLease/';

    private static self $instance;

    private string $auth_token = '';

    private string $default_location_code = '';

    public static function get_instance(): self
    {
        if (empty(self::$instance)) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    private function __construct()
    {
        $option = (array) \WPCF7::get_option(self::service_name, []);

        if (isset($option['auth_token'])) {
            $this->auth_token = $option['auth_token'];
        }

        if (isset($option['default_location_code'])) {
            $this->default_location_code = $option['default_location_code'];
        }
    }

    public function get_title()
    {
        return 'SiteLink';
    }

    public function is_active(): bool
    {
        return $this->get_auth_token() !== '';
    }

    public function get_auth_token()
    {
        return $this->auth_token;
    }

    public function get_default_location_code()
    {
        return $this->default_location_code;
    }

    public function get_categories()
    {
        return [
            'conversion_tracking',
        ];
    }

    public function icon()
    {
        $svg = <<<XML
		<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 17.94 15.55">
			<path fill="currentColor" d="M13.46,0h-9L0,7.78l4.49,7.77h9l4.48-7.77ZM8.16,7.31H2.11L5.15,2ZM6,1.6H7.49l6.15,10.51-.8,1.4Zm6.53,6.71h3.28l-1.62,2.82Zm-.58-1L8.63,1.6h3.91l3.29,5.71Zm-3.24,1L12,14H5.4L2.11,8.31Z"/>
		</svg>
		XML;

        $src = esc_attr("data:image/svg+xml;utf8,{$svg}");

        echo <<<HTML
		<img class="icon" src="$src" style="height: 36px; width: auto;" />
		HTML;
    }

    public function link()
    {
        return '';
    }

    protected function menu_page_url($args = '')
    {
        $args = wp_parse_args($args, []);

        $url = menu_page_url('wpcf7-integration', false);
        $url = add_query_arg(['service' => static::service_name], $url);

        if (!empty($args)) {
            $url = add_query_arg($args, $url);
        }

        return $url;
    }

    protected function save_data()
    {
        \WPCF7::update_option(static::service_name, [
            'auth_token' => $this->auth_token,
            'default_location_code' => $this->default_location_code,
        ]);
    }

    protected function reset_data()
    {
        $this->auth_token = '';
        $this->default_location_code = '';
        $this->save_data();
    }

    public function load($action = '')
    {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handle_post($action);
        }
    }

    protected function handle_post($action)
    {
        switch ($action) {
            case 'setup':
                check_admin_referer($this->get_nonce_name($action));

                if (!empty($_POST['reset'])) {
                    $this->reset_data();
                    wp_safe_redirect($this->menu_page_url(['action' => 'setup']));
                    exit;
                }

                $auth_token = trim($_POST['auth_token'] ?? '');
                $default_location_code = trim($_POST['default_location_code'] ?? '');

                if (!$auth_token) {
                    wp_safe_redirect($this->menu_page_url(['action' => 'setup', 'message' => 'missing']));
                    exit;
                }

                if (!$this->check_auth_token($auth_token)) {
                    wp_safe_redirect($this->menu_page_url(['action' => 'setup', 'message' => 'invalid']));
                    exit;
                }

                $this->auth_token = $auth_token;
                $this->default_location_code = $default_location_code;

                $this->save_data();

                $redirect_to = $this->menu_page_url(['message' => 'success']);

                wp_safe_redirect($redirect_to);
                exit;
        }
    }

    public function admin_notice($message = '')
    {
        if ('missing' == $message) {
            echo sprintf(
                '<div class="notice notice-error"><p><strong>%1$s</strong>: %2$s</p></div>',
                esc_html(__('Error', 'contact-form-7')),
                esc_html(__('Auth token is required.', 'contact-form-7'))
            );
        }

        if ('invalid' == $message) {
            echo sprintf(
                '<div class="notice notice-error"><p><strong>%1$s</strong>: %2$s</p></div>',
                esc_html(__('Error', 'contact-form-7')),
                esc_html(__('Invalid auth token.', 'contact-form-7'))
            );
        }

        if ('success' == $message) {
            echo sprintf(
                '<div class="notice notice-success"><p>%s</p></div>',
                esc_html(__('Settings saved.', 'contact-form-7'))
            );
        }
    }

    public function display($action = '')
    {
        echo sprintf(
            '<p>%s</p>',
            <<<HTML
			SiteLink’s self-storage software helps small and large self-storage operators manage their self-storage tenants and grow profits while reducing costs in 2017.
			HTML
        );

        // echo sprintf(
        //     '<p><strong>%s</strong></p>',
        //     wpcf7_link(
        //         __( 'https://example.com/', 'contact-form-7' ),
        //         __( 'SiteLink integration', 'contact-form-7' )
        //     )
        // );

        if ($this->is_active()) {
            echo sprintf(
                '<p class="dashicons-before dashicons-yes">%s</p>',
                esc_html(__("SiteLink is connected.", 'contact-form-7'))
            );
        }

        if ($action === 'setup') {
            $this->display_setup();
        } else {
            echo sprintf(
                '<p><a href="%1$s" class="button">%2$s</a></p>',
                esc_url($this->menu_page_url(['action' => 'setup'])),
                esc_html(__('Setup Integration', 'contact-form-7'))
            );
        }
    }

    private function display_setup()
    {
        $auth_token = $this->get_auth_token();
        $default_location_code = $this->default_location_code;

        ?>
        <form method="post" action="<?php echo esc_url($this->menu_page_url(['action' => 'setup'])); ?>">
            <?php wp_nonce_field($this->get_nonce_name('setup')); ?>
            <table class=" form-table">
                <tbody>
                    <tr>
                        <th scope="row"><label for="auth_token">
                                <?php echo esc_html(__('Auth token', 'contact-form-7')); ?>
                            </label></th>
                        <td>
                            <?php
                            if ($this->is_active()) {
                                echo esc_html(wpcf7_mask_password($auth_token));
                                echo sprintf(
                                    '<input type="hidden" value="%s" id="auth_token" name="auth_token" />',
                                    esc_attr($auth_token)
                                );
                            } else {
                                echo sprintf(
                                    '<input type="text" aria-required="true" value="%s" id="auth_token" name="auth_token" class="regular-text code" required />',
                                    esc_attr($auth_token)
                                );
                                echo <<<HTML
								<p class="description">
									The bearer token for the R6 Lead to Lease API.
								</p>
								HTML;
                            }
                            ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="default_location_code">
                                <?php echo esc_html(__('Default location code', 'contact-form-7')); ?>
                            </label></th>
                        <td>
                            <?php
                            if ($this->is_active()) {
                                echo esc_html($default_location_code);
                                echo sprintf(
                                    '<input type="hidden" value="%s" id="default_location_code" name="default_location_code" />',
                                    esc_attr($default_location_code)
                                );
                            } else {
                                echo sprintf(
                                    '<input type="text" aria-required="true" value="%s" id="default_location_code" name="default_location_code" class="regular-text" />',
                                    esc_attr($default_location_code)
                                );
                                echo <<<HTML
								<p class="description">
									The location code can be set on individual contact forms.
								</p>
								HTML;
                            }
                            ?>
                        </td>
                    </tr>
                </tbody>
            </table>
            <?php
            if ($this->is_active()) {
                submit_button(
                    __('Remove Auth Token', 'contact-form-7'),
                    'small',
                    'reset',
                    true,
                    [
                        'onclick' => 'return confirm("' . esc_js(__("Are you sure you want to remove the auth token?", 'contact-form-7')) . '")',
                    ]
                );
            } else {
                submit_button(__('Save Changes', 'contact-form-7'));
            }
            ?>
        </form>
        <?php
    }

    private function get_nonce_name(string $action): string
    {
        $service = static::service_name;

        return "wpcf7-{$service}-{$action}";
    }

    private function check_auth_token(string $auth_token): bool
    {
        if (!$auth_token) {
            return false;
        }

        try {
            $old_token = $this->auth_token;
            $this->auth_token = $auth_token;

            // The API doesn't actually have a "CheckToken" route
            // But since all requests get authenticated before routing, this fake route should work fine
            $response = $this->api_get('CheckToken');
        } finally {
            $this->auth_token = $old_token;
        }

        if ($response['response']['code'] !== 200 || $response['body']['error']['code'] === 401) {
            return false;
        }

        return true;
    }

    public function api_add_inquiry(array $data)
    {
        $data['unit_name'] = 'WEBINQ';
        $data['source'] = 'Contact form on ' . get_bloginfo('name');
        $data['type'] = 'lead';

        $response = $this->api_post('AddInquiry', $data);

        return wp_remote_retrieve_body($response);
    }

    private function api_get(string $endpoint, array $data = null)
    {
        if (!empty($data)) {
            $endpoint .= '?' . http_build_query($data);
        }

        return $this->api_request('GET', $endpoint);
    }

    private function api_post(string $endpoint, array $data = null)
    {
        return $this->api_request('POST', $endpoint, $data);
    }

    private function api_request(string $method, string $endpoint, array $data = null)
    {
        $headers = [
            'Authorization' => "Bearer {$this->auth_token}",
        ];

        if (!empty($data)) {
            $filteredData = apply_filters('r6_kitchen_sink_sitelink_api_request_data', $data, $method, $endpoint);
            $data = json_encode($filteredData);
            $headers['Content-Type'] = 'application/json';
        }
        
        $response = wp_remote_request(self::service_url . $endpoint, [
            'method' => $method,
            'headers' => $headers,
            'body' => $data,
        ]);

        if (is_wp_error($response)) {
            $error_code = null;
            if (!empty($response->get_error_code())) {
                $error_code = $response->get_error_code();
            }
            throw new \Exception($response->get_error_message(), $error_code);
        }

        $response['body'] = json_decode($response['body'], true);

        return $response;
    }
}
