<?php

namespace R6Digital\KitchenSink\Health;

class MissingMedia404
{
    public function __construct()
    {
        add_filter('site_status_tests', [$this, 'add_tests']);
        add_action('rest_api_init', [$this, 'register_rest_routes']);
        add_action('template_redirect', [$this, 'add_header_to_media_404']);
    }

    public function add_tests($tests)
    {
        $tests['async']['bypass_wordpress_media_404'] = [
            'label' => 'WordPress is generating 404 pages for missing media',
            'test' => rest_url('r6-kitchen-sink/v1/health/bypass-wordpress-media-404'),
            'has_rest' => true,
            'skip_cron' => true,
        ];

        return $tests;
    }

    public function add_header_to_media_404()
    {
        if (!is_404()) {
            return;
        }

        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $filename = basename($path);

        if (!preg_match('/^r6-kitchen-sink-health-check/', $filename)) {
            return;
        }

        header('X-R6-Kitchen-Sink-Health-Check: 1');
    }

    public function register_rest_routes()
    {
        register_rest_route(
            'r6-kitchen-sink/v1',
            '/health/bypass-wordpress-media-404',
            [
                'methods' => 'GET',
                'callback' => [$this, 'check_wordpress_media_404'],
                'permission_callback' => function () {
                    return current_user_can('view_site_health_checks');
                },
            ],
        );
    }

    public function check_wordpress_media_404()
    {
        global $is_apache, $is_nginx;

        $uploads = wp_upload_dir(null, false, false);
        $unique_filename = wp_unique_filename($uploads['basedir'], 'r6-kitchen-sink-health-check.jpg');
        $response = wp_remote_head($uploads['baseurl'] . '/' . $unique_filename);

        $result = [
            'label' => 'The web server is handling the 404 pages for missing media',
            'status' => 'good',
            'badge' => [
                'label' => __('Performance'),
                'color' => 'blue',
            ],
            'description' => <<<HTML
                <p>WordPress should not generate 404 pages for missing media items, the web server (Apache, NGINX, etc) should handle the response instead.</p>
                HTML,
            'actions' => '',
            'test' => 'bypass_wordpress_media_404',
        ];

        if (is_wp_error($response)) {
            $result['status'] = 'critical';
            $result['label'] = 'Unable to check if the web server is handling the 404 pages for missing media';
            $result['description'] = sprintf('<p>%s</p>', $response->get_error_message());

            return $result;
        }

        if (wp_remote_retrieve_response_code($response) !== 404) {
            $result['status'] = 'critical';
            $result['label'] = 'Did not receive a 404 response for missing media';
            $result['description'] = sprintf('<p>Expected a 404 response, receive a %s response.</p>', wp_remote_retrieve_response_code($response));

            return $result;
        }

        if (!empty(wp_remote_retrieve_headers($response)['x-r6-kitchen-sink-health-check'])) {
            $w3tc_settings_page = admin_url('admin.php?page=w3tc_browsercache');
            $result['status'] = 'critical';
            $result['label'] = 'WordPress is handling the 404 pages for missing media';

            $filetypes = 'css|js|avi|avif|avifs|bmp|eot|gif|ico|jpg|jpeg|jpe|webp|json|mid|midi|mov|qt|mp3|m4a|mp4|m4v|mpeg|mpg|mpe|webm|otf|ogg|ogv|png|svg|svgz|tif|tiff|ttf|wav|wma|woff|woff2';
            $code_style = 'padding: 0.75em 1em; border: 1px solid #e1e1e1; background-color: #f7f7f7;';

            if (defined('W3TC')) {
                $result['actions'] = <<<HTML
                <p><strong>Instructions for W3 Total Cache</strong></p>
                <ol>
                    <li>Go to <a href="{$w3tc_settings_page}">Performance > Browser Cache</a></li>
                    <li>Enable the option <em>Do not process 404 errors for static objects with WordPress</em></li>
                </ol>
                HTML;
            } else if ($is_apache) {
                $htaccess_rules = <<<HTACCESS
                <IfModule mod_rewrite.c>
                RewriteEngine On
                RewriteCond %{REQUEST_FILENAME} !-f
                RewriteCond %{REQUEST_FILENAME} !-d
                RewriteCond %{REQUEST_URI} \.($filetypes)\$ [NC]
                RewriteRule .* - [L]
                </IfModule>
                HTACCESS;
                $htaccess_rules = esc_html($htaccess_rules);

                $result['actions'] = <<<HTML
                <p><strong>Configure Apache</strong></p>
                <p>Add the following lines to the top of the <code>.htaccess</code> file:</p>
                <pre class="pre" style="{$code_style}">{$htaccess_rules}</pre>
                HTML;
            } else if ($is_nginx) {
                $nginx_rules = <<<NGINX
                location ~* \.($filetypes)\$ {
                    try_files \$uri =404;
                    access_log off;
                    log_not_found off;
                }
                NGINX;
                $nginx_rules = esc_html($nginx_rules);

                $result['actions'] = <<<HTML
                <p><strong>Configure NGINX</strong></p>
                <p>Add the following lines to the <code>server</code> block before any other <code>location</code>:</p>
                <pre class="pre" style="{$code_style}">{$nginx_rules}</pre>
                HTML;
            }

            return $result;
        }

        return $result;
    }
}
