<?php

namespace R6Digital\KitchenSink\WPCF7;

class SitelinkFormTab extends FormTab
{
    private static self $instance;

    public static function get_instance(): self
    {
        if (empty(self::$instance)) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    public function get_service()
    {
        return SitelinkService::get_instance();
    }

    public function is_active(): bool
    {
        return $this->get_service()->is_active();
    }

    public function get_name(): string
    {
        return 'sitelink';
    }

    public function get_title(): string
    {
        return 'SiteLink';
    }

    public function get_defaults(): array
    {
        return [
            'enabled' => false,
            'location_code' => '',
            'title' => '',
            'first_name' => '',
            'last_name' => '',
            'company' => '',
            'address_line_1' => '',
            'address_line_2' => '',
            'suburb' => '',
            'state' => '',
            'postcode' => '',
            'phone_number' => '',
            'mobile_number' => '',
            'email' => '',
            'date_needed' => '',
            'length_of_time' => '',
            'storage_needs' => '',
            'comment' => '',
            'date_of_birth' => '',
            'marketing_source' => '',
            'marketing_distance' => '',
            'marketing_reason' => '',
            'marketing_customer_type' => '',
            'marketing_what_is_stored' => '',
            'marketing_why_this_facility' => '',
        ];
    }

    public function get_submitted(): array
    {
        $fields = isset($_POST['wpcf7-sitelink'])
            ? (array) $_POST['wpcf7-sitelink']
            : [];

        return $fields;
    }

    public function get_values(\WPCF7_ContactForm $form)
    {
        return wp_parse_args(
            $form->prop($this->get_service()::service_name),
            $this->get_defaults(),
        );
    }

    protected function get_fields(array $values): array
    {
        $fields = [
            'enabled' => ['label' => 'Enable integration', 'type' => 'checkbox'],
            'location_code' => ['label' => 'Location Code'],
            'title' => ['label' => 'Title'],
            'first_name' => ['label' => 'First Name'],
            'last_name' => ['label' => 'Last Name', 'required' => true],
            'company' => ['label' => 'Company'],
            'address_line_1' => ['label' => 'Address Line 1'],
            'address_line_2' => ['label' => 'Address Line 2'],
            'suburb' => ['label' => 'Suburb'],
            'state' => ['label' => 'State'],
            'postcode' => ['label' => 'Postcode'],
            'phone_number' => ['label' => 'Phone Number'],
            'mobile_number' => ['label' => 'Mobile Number'],
            'email' => ['label' => 'Email'],
            'date_needed' => ['label' => 'Date Needed'],
            'length_of_time' => ['label' => 'Length Of Time'],
            'storage_needs' => ['label' => 'Storage Needs'],
            'comment' => ['label' => 'Comment', 'type' => 'textarea'],
            'date_of_birth' => ['label' => 'Date Of Birth'],
            'marketing_source' => ['label' => 'Marketing: Source'],
            'marketing_distance' => ['label' => 'Marketing: Distance'],
            'marketing_reason' => ['label' => 'Marketing: Reason'],
            'marketing_customer_type' => ['label' => 'Marketing: Customer Type'],
            'marketing_what_is_stored' => ['label' => 'Marketing: What Is Stored'],
            'marketing_why_this_facility' => ['label' => 'Marketing: Why This Facility'],
        ];

        foreach ($fields as $slug => &$field) {
            $field = wp_parse_args($field, [
                'slug' => $slug,
                'id' => "wpcf7-sitelink-$slug",
                'name' => "wpcf7-sitelink[$slug]",
                'type' => 'text',
                'label' => $slug,
                'required' => false,
                'help' => '',
                'placeholder' => '',
                'value' => $values[$slug] ?? '',
            ]);
        }
        unset($field);

        if (($default_location_code = $this->get_service()->get_default_location_code())) {
            $fields['location_code']['placeholder'] = $default_location_code;
            $fields['location_code']['help'] = "Leave blank to use default value (<code>{$default_location_code}</code>)";
        } else {
            $fields['location_code']['required'] = 'required';
        }

        return $fields;
    }

    public function render(): void
    {
        $form = \WPCF7_ContactForm::get_current();
        $tags = $form->collect_mail_tags();

        $values = $this->get_values($form);
        $fields = $this->get_fields($values);

        // Pull the checkbox out of the table
        $enabled_field = $fields['enabled'];
        unset($fields['enabled']);

        ?>
        <h2>SiteLink - Lead to Lease</h2>

        <p>This integration creates a lead in SiteLink when this form is submitted.</p>

        <p>
            <label for="<?= esc_attr($enabled_field['id']) ?>">
                <?php $this->render_input($enabled_field) ?>
                <?= esc_html($enabled_field['label']) ?>
            </label>
        </p>

        <fieldset <?= $values['enabled'] ? '' : 'hidden' ?>>
            <legend>
                Populate the fields below with static values, or use the mail tags.

                <div class="mailtags">
                    <?php foreach ($tags as $tag) {
                        echo sprintf(
                            ' <span class="mailtag code">[%s]</span> ',
                            esc_html($tag)
                        );
                    } ?>
                </div>
            </legend>

            <table class="form-table">
                <?php foreach ($fields as $field) { ?>
                    <tr>
                        <th>
                            <label for="<?= esc_attr($field['id']) ?>">
                                <?= esc_html($field['label']) ?>
                                <?php if ($field['required']) { ?>
                                    <span class="required">*</span>
                                <?php } ?>
                            </label>
                        </th>
                        <td>
                            <?php $this->render_input($field) ?>

                            <?php if ($field['help']) { ?>
                                <p class="description">
                                    <?= $field['help'] ?>
                                </p>
                            <?php } ?>
                        </td>
                    </tr>
                <?php } ?>
            </table>
        </fieldset>

        <script type="module">
            const $panel = jQuery( '#sitelink-panel' );

            jQuery( '#wpcf7-sitelink-enabled', $panel )
                .on( 'click', e => jQuery( '> fieldset', $panel ).attr( 'hidden', !e.target.checked ) );

            jQuery( '.mailtag', $panel )
                .on( 'click', function () {
                    const range = document.createRange();
                    range.selectNodeContents( this );
                    window.getSelection().addRange( range );
                } )
        </script>
        <?php
    }

    private function render_input(array $field)
    {
        $id = esc_attr($field['id']);
        $name = esc_attr($field['name']);
        $type = esc_attr($field['type']);
        $placeholder = esc_attr($field['placeholder']);
        $value = esc_attr($field['value']);

        switch ($type) {
            default:
                echo <<<HTML
				<input type="{$type}" name="{$name}" id="{$id}" placeholder="{$placeholder}" value="{$value}" class="large-text code">
				HTML;
                break;

            case 'checkbox':
                $checked = $field['value'] ? 'checked' : '';
                echo <<<HTML
				<input type="{$type}" name="{$name}" id="{$id}" value="1" {$checked}>
				HTML;
                break;

            case 'textarea':
                $value = esc_html($field['value'] ?? '');
                echo <<<HTML
				<textarea name="{$name}" id="{$id}" cols="100" rows="4" placeholder="{$placeholder}" class="large-text code">{$value}</textarea>
				HTML;
                break;
        }
    }

    function mail_sent(\WPCF7_ContactForm $form)
    {
        $values = $this->get_values($form);

        if (!$values['enabled']) {
            return;
        }

        $payload = [
            'location_code' => $values['location_code'],
            'title' => $values['title'],
            'first_name' => $values['first_name'],
            'last_name' => $values['last_name'],
            'company' => $values['company'],
            'address_line_1' => $values['address_line_1'],
            'address_line_2' => $values['address_line_2'],
            'suburb' => $values['suburb'],
            'state' => $values['state'],
            'postcode' => $values['postcode'],
            'phone_number' => $values['phone_number'],
            'mobile_number' => $values['mobile_number'],
            'email' => $values['email'],
            'date_needed' => $values['date_needed'],
            'length_of_time' => $values['length_of_time'],
            'storage_needs' => $values['storage_needs'],
            'comment' => $values['comment'],
            'date_of_birth' => $values['date_of_birth'],
            'marketing_source' => $values['marketing_source'],
            'marketing_distance' => $values['marketing_distance'],
            'marketing_reason' => $values['marketing_reason'],
            'marketing_customer_type' => $values['marketing_customer_type'],
            'marketing_what_is_stored' => $values['marketing_what_is_stored'],
            'marketing_why_this_facility' => $values['marketing_why_this_facility'],
        ];

        foreach ($payload as &$value) {
            $value = wpcf7_mail_replace_tags($value);
        }
        unset($value);

        if (empty($payload['location_code'])) {
            $payload['location_code'] = $this->get_service()->get_default_location_code();
        }

        if ($payload['date_of_birth']) {
            $payload['date_of_birth'] = date('Y-m-d', strtotime($payload['date_of_birth']));
        }

        if ($payload['date_needed']) {
            $payload['date_needed'] = date('Y-m-d', strtotime($payload['date_needed']));
        }

        if (!$payload['location_code']) {
            $this->log('Skipping submission because location code is not filled.', $form);
            return;
        }

        if (!$payload['last_name']) {
            $this->log('Skipping submission because last name is not filled.', $form);
            return;
        }

        try {
            $response = $this->get_service()->api_add_inquiry($payload);

            $this->log('API response: ' . json_encode($response), $form);
        } catch (\Throwable $e) {
            $this->log('API error: ' . $e->getMessage(), $form);
        }
    }

    protected function log(string $message, ?\WPCF7_ContactForm $form)
    {
        $prefix = '[R6 Kitchen Sink] SiteLink Lead to Lease integration';

        if ($form) {
            $form_id = $form->id();
            $form_name = $form->name();
            $prefix .= " on form '{$form_name}' (ID {$form_id})";
        }

        @error_log("{$prefix}: {$message}");
    }
}
